#!/bin/sh
#
# Authors: Franz-Peter Glawar, 2012
#
# Installation fuer OPList
#
# Bei der Installation koennen folgende Parameter angegeben werden:
# --install              fuehrt die Installation durch
# --help                 zeigt die Hilfe an
# --dbtype DBTYPE        DBTYPE ist entweder oracle oder mysql
# --schema SCHEMANAME    benutzt SCHEMANAME als Schema
# --sid LMS              benutzt LMS als SID
# --dbname DBNAME        name der MySQL-Datenbank
# --host HOST            Hostname, wo die MySQL-Datenbank läuft
# --user USERNAME        Benutzername für die MySQL-Datenbank
# --passwd PASSWORD      Kennwort für die MySQL-Datenbank
# --target /www/htdocs   benutzt /www/htdocs als Targetverzeichnis
# --group www            benutzt www als Gruppe
#
######################################################################
# v2011-11-29 Paco erste Installations-Version
# v2011-11-30 Paco Erweiterung, so dass nicht irrtuemlich ohne Parameter
#                  installiert wird --> --install-Parameter notwendig
# v2014-07-21 Paco Erweiterung, dass ein Update-Skript erzeugt wird
# v2014-07-25 Paco Erweiterung des Installationsscripts für Installation
#                  bzw. Update auch für MySQL
# v2014-08-04 Paco Berechtigung passt noch nicht ganz
# v2014-11-24 Paco neuer Conf-Parameter EmailFontSize
# 2015-01-10 Paco neues Installationspaket, wo nur ganz am Anfang das Datenmodell
#                 installiert wird, der Rest wird danach vom PHP-File selbst
#                 aktualisiert
# 2015-01-16 Paco Installationspaket schreibt nun automatisch die
#                 zu verwendende Datenbank in die Konfigurationsdatei
# 2015-02-01 Paco oplist.js, oplist.css und sort.js werden aus dem
#                 root-Verzeichnis von OPList geloescht
# 2015-08-09 Paco Installation beruecksichtigt indiv-php-Seiten
######################################################################
TARGETDIR=/srv/www/htdocs
APPDIR=/u01/app
ZIPFILE=oplist.tgz
DBTYPE=unknown
DBHOST=localhost
SCHEMA=SYSADM
SID=LMS
USER=root
PASSWD=unbekannt
GROUP=www
# DEVELOPMENT=1 bewirkt, dass keine Daten in die DB geschrieben werden
# DEVELOPMENT=0 fuehrt die Installation aus
DEVELOPMENT=0
DEBUG=0
INSTALLYES=0
WITH_DB_MODEL=1
######################################################################


#---------------------------------------------------------------------
# Ueberprueft, ob der angemeldete Benutzer ROOT ist
#---------------------------------------------------------------------
checkIfRoot() {
    if [ ! "`id -u`" -eq 0 ]; then
        echo "Du bist leider nicht root"
        exit 1
    fi
}


#---------------------------------------------------------------------
# zeigt Beispiele an
#---------------------------------------------------------------------
showExamples () {
    echo "Installation auf Apache (Oracle):"
    echo "install.sh --install --dbtype oracle --schema OPLIST --sid LMS --target /srv/www/htdocs --group www"
    echo
    echo "Installation auf Apache (MySQL):"
    echo "install.sh --install --dbtype mysql --dbname OPLIST --host localhost --user username --passwd password --target /srv/www/htdocs --group www"
    echo
    echo "Installation auf XAmp (Oracle):"
    echo "install.sh --install --dbtype oracle --schema OPLIST --sid LMS --target /opt/lampp/htdocs --group root"
    echo
    echo "Installation auf XAmp (MySQL):"
    echo "install.sh --install --dbtype mysql --dbname OPLIST --host localhost --user username --passwd password --target /opt/lampp/htdocs --group root"
    echo
    echo "   --help                 zeigt die Hilfe an"
    echo
    echo "   Allgemein"
    echo "   --install              fuehrt die Installation durch"
    echo "   --dbtype DBTYPE        DBTYPE ist entweder oracle oder mysql"
    echo "   --target /www/htdocs   benutzt /www/htdocs als Targetverzeichnis"
    echo "   --group www            benutzt www als Gruppe"
    echo
    echo "  Oracle:"
    echo "   --schema SCHEMANAME    benutzt SCHEMANAME als Schema"
    echo "   --sid LMS              benutzt LMS als SID"
    echo
    echo "  MySQL:"
    echo "   --dbname DBNAME        name der MySQL-Datenbank"
    echo "   --host HOST            Hostname, wo die MySQL-Datenbank läuft"
    echo "   --user USERNAME        Benutzername für die MySQL-Datenbank"
    echo "   --passwd PASSWORD      Kennwort für die MySQL-Datenbank"
}


#---------------------------------------------------------------------
# Ueberprueft die angegebenen Parameter
#---------------------------------------------------------------------
checkParams () {
    READOPT=0
    UPDATEYES=0
    while [ ${READOPT} -eq 0 ]; do
        case ${1} in
            "--install")
                INSTALLYES=1
                shift
                ;;
            "--update")
                INSTALLYES=1
                UPDATEYES=1
                shift
                ;;
            "--schema" | "--user")
                SCHEMA=${2}
                shift
                shift
                ;;
            "--passwd")
                PASSWD=${2}
                shift
                shift
                ;;
            "--host")
                DBHOST=${2}
                shift
                shift
                ;;
            "--sid" | "--dbname")
                SID=${2}
                shift
                shift
                ;;
            "--target")
                TARGETDIR=${2}
                shift
                shift
                ;;
            "--group")
                GROUP=${2}
                shift
                shift
                ;;
            "--dbtype")
                DBTYPE=${2}
                shift
                shift
                ;;
            "--debug")
                DEBUG=1
                shift
                ;;
            *)
                # if user option starts with '-'
                if [ ! "`echo $1 | grep -s '^-'`" = "" ] ; then
                    echo "unbekannte option $1"
                    exit 2
                else
                    # reading of users options finished
                    READOPT=1
                fi
        esac
    done
}


askUserIfInstallationShouldStart () {
    echo "Soll die Installation fortgesetzt werden? [ J | n ]"

    read ANSWER
    if [ -z ${ANSWER} ]; then
        return 0
    else
        case ${ANSWER} in
            j* | J* | y* | Y*)
                return 0
                ;;
            *)
                return 1
                ;;
        esac
    fi
}


#---------------------------------------------------------------------
# Main
#---------------------------------------------------------------------
main() {
    checkIfRoot

    if [ "${1}" = "--help" ]; then
        showExamples
        exit 0
    fi

    checkParams $*
    if [ ${INSTALLYES} -eq 0 ]; then
        showExamples
        exit 0
    fi

    if [ ${DEVELOPMENT} -eq 1 ]; then
        echo "Update <${UPDATEYES}>"
        echo
    fi

    POSSIBLE_ERROR=1
    if [ ${DBTYPE} = "oracle" ]; then
        POSSIBLE_ERROR=0
        if [ ${DEVELOPMENT} -eq 1 ]; then
            echo "dbtype <${DBTYPE}>"
            echo "schema <${SCHEMA}>"
            echo "sid    <${SID}>"
            echo "target <${TARGETDIR}>"
            echo "group  <${GROUP}>"
        fi
    fi
    if [ ${DBTYPE} = "mysql" ]; then
        POSSIBLE_ERROR=0
        if [ ${DEVELOPMENT} -eq 1 ]; then
            echo "dbtype   <${DBTYPE}>"
            echo "host     <${DBHOST}>"
            echo "dbname   <${SID}>"
            echo "user     <${SCHEMA}>"
            echo "password <${PASSWD}>"
            echo "target   <${TARGETDIR}>"
            echo "group    <${GROUP}>"
        fi
    fi
    echo

    if [ ${POSSIBLE_ERROR} -eq 1 ]; then
        echo "Keine gueltige Datenbank spezizifiert (${DBTYPE})."
        exit 1
    fi

    if [ ${DEBUG} -eq 0 ]; then
        UPDATESCRIPT=update.sh
        DEBUGTEXT=""
    else
        UPDATESCRIPT=updateDebug.sh
        DEBUGTEXT="--debug"
    fi

    echo "Es werden folgende Parameter verwendet:"
    if [ ${UPDATEYES} -eq 1 ]; then
        INSTALLORUPDATE="--update"
    else
        INSTALLORUPDATE="--install"
    fi
    if [ ${DBTYPE} = "oracle" ]; then
        echo "install.sh ${INSTALLORUPDATE} --dbtype ${DBTYPE} --schema ${SCHEMA} --sid ${SID} --target ${TARGETDIR} --group ${GROUP} ${DEBUGTEXT}"
    else
        echo "install.sh ${INSTALLORUPDATE} --dbtype ${DBTYPE} --dbname ${SID} --host ${DBHOST} --user ${SCHEMA} --passwd ${PASSWD} --target ${TARGETDIR} --group ${GROUP} ${DEBUGTEXT}"
    fi
    echo " "

    if [ ${UPDATEYES} -eq 0 ]; then
        askUserIfInstallationShouldStart
    fi

    if [ $? -eq 0 ]; then
        echo "Installation wird durchgeführt"
    else
        echo "Abbruch der Installation durch den Benutzer"
        exit 1
    fi

    echo "Erzeuge das Update-Script <${UPDATESCRIPT}>"
    echo "#!/bin/sh" > ${UPDATESCRIPT}
    echo "#" >> ${UPDATESCRIPT}
    echo "# Update-Script fuer OPList" >> ${UPDATESCRIPT}
    echo "#" >> ${UPDATESCRIPT}
    if [ ${DBTYPE} = "oracle" ]; then
        echo "./install.sh --update --dbtype ${DBTYPE} --schema ${SCHEMA} --sid ${SID} --target ${TARGETDIR} --group ${GROUP} ${DEBUGTEXT}" >> ${UPDATESCRIPT}
    else
        echo "./install.sh --update --dbtype ${DBTYPE} --dbname ${SID} --host ${DBHOST} --user ${SCHEMA} --passwd ${PASSWD} --target ${TARGETDIR} --group ${GROUP} ${DEBUGTEXT}" >> ${UPDATESCRIPT}
    fi
    chmod 755 ${UPDATESCRIPT}

    # kopieren der Dateien
    echo "Kopiere <${ZIPFILE}> nach <${TARGETDIR}> und wechsle nach <${TARGETDIR}>"
    if [ ${DEBUG} -eq 0 ]; then
        cp ${ZIPFILE} ${TARGETDIR}
        cd ${TARGETDIR}
    fi

    # falls index.php bereits existiert (update) --> Sicherung
    if [ -f  ${TARGETDIR}/oplist/index.php ]; then
        GLATEMPTIMESTAMP=`date +%Y_%m_%d-%H_%M_%S`
        echo "Sichere <${TARGETDIR}/oplist/index.php> um nach <${TARGETDIR}/oplist/index.php_${GLATEMPTIMESTAMP}>"
        if [ ${DEBUG} -eq 0 ]; then
            cp ${TARGETDIR}/oplist/index.php ${TARGETDIR}/oplist/index.php_${GLATEMPTIMESTAMP}
        fi
    fi

    # falls include/update.php bereits existiert --> Sicherung
    if [ -f  ${TARGETDIR}/oplist/include/update.php ]; then
        GLATEMPTIMESTAMP=`date +%Y_%m_%d-%H_%M_%S`
        echo "Sichere <${TARGETDIR}/oplist/include/update.php> um nach <${TARGETDIR}/oplist/include/update.php_${GLATEMPTIMESTAMP}>"
        if [ ${DEBUG} -eq 0 ]; then
            cp ${TARGETDIR}/oplist/include/update.php ${TARGETDIR}/oplist/include/update.php_${GLATEMPTIMESTAMP}
        fi
    fi

    # falls oplist.js im root-Verzeichnis existiert (alte Version)
    # wird diese geloescht
    if [ -f  ${TARGETDIR}/oplist/oplist.js ]; then
        rm ${TARGETDIR}/oplist/oplist.js
    fi

    # falls sort.js im root-Verzeichnis existiert (alte Version)
    # wird diese geloescht
    if [ -f  ${TARGETDIR}/oplist/sort.js ]; then
        rm ${TARGETDIR}/oplist/sort.js
    fi

    # falls oplist.css im root-Verzeichnis existiert (alte Version)
    # wird diese geloescht
    if [ -f  ${TARGETDIR}/oplist/oplist.css ]; then
        rm ${TARGETDIR}/oplist/oplist.css
    fi

    # extrahieren der Dateien
    echo "Extrahiere <${ZIPFILE}>"
    if [ ${DEBUG} -eq 0 ]; then
        tar xfvz ${ZIPFILE}
    fi

    # falls include/index_indiv.php nicht existiert --> Template umbenennen
    if [ ! -f  ${TARGETDIR}/oplist/include/index_indiv.php ]; then
        # wir benennen das Template um
        echo "Benenne Vorlage <${TARGETDIR}/oplist/include/index_indiv.php.template> um nach <${TARGETDIR}/oplist/include/index_indiv.php>"
        if [ ${DEBUG} -eq 0 ]; then
            cp ${TARGETDIR}/oplist/include/index_indiv.php.template ${TARGETDIR}/oplist/include/index_indiv.php
        fi
    fi
    chmod g+w,o+w ${TARGETDIR}/oplist/include/index_indiv.php

    # Rechte setzen
    echo "Recht <root:${GROUP}> für <oplist>"
    if [ ${DEBUG} -eq 0 ]; then
        chown -R root:${GROUP} oplist
    fi

    # Archiv loeschen
    echo "Lösche Archiv <${ZIPFILE}>"
    if [ ${DEBUG} -eq 0 ]; then
        rm ${ZIPFILE}
    fi

    # index.php fuer jedermann aenderbar machen
    echo "Setze Berechtigung für <index.php> und <uploads>"
    if [ ${DEBUG} -eq 0 ]; then
        cd oplist
        chmod g+r index.php
        chmod g+w index.php
        chmod o+r index.php
        chmod o+w index.php
        chmod 777 uploads
    fi

    # Rechte setzen für log und conf
    echo "Recht <root:${GROUP}> für <conf> und <log>"
    if [ ${DEBUG} -eq 0 ]; then
        chmod -R o+r,o+w conf
        chmod -R o+r,o+w log
    fi

    # ueberpruefe ob es schon ein oplist.ini gibt
    if [ -f ${TARGETDIR}/oplist/conf/oplist.ini ]; then
        echo "${TARGETDIR}/oplist/conf/oplist.ini existiert bereits"
    else
        echo "Bennene <${TARGETDIR}/oplist/conf/oplist.ini.template> nach <${TARGETDIR}/oplist/conf/oplist.ini>"
        if [ ${DEBUG} -eq 0 ]; then
            cp ${TARGETDIR}/oplist/conf/oplist.ini.template ${TARGETDIR}/oplist/conf/oplist.ini
        fi
    fi
    chmod g+w,o+w ${TARGETDIR}/oplist/conf/oplist.ini

    # link in /u01/app/oplist erstellen (fuer einfacheres editieren)
    echo "Verlinke <index.php> und <oplist.ini> nach <${APPDIR}/oplist>"
    if [ ${DEBUG} -eq 0 ]; then
        if [ ! -d ${APPDIR}/oplist ]; then
            mkdir ${APPDIR}/oplist
        fi
        cd ${APPDIR}/oplist
        ln -sf ${TARGETDIR}/oplist/index.php .
        ln -sf ${TARGETDIR}/oplist/conf/oplist.ini .
        ln -sf ${TARGETDIR}/oplist/log/oplist.log .
    fi

    echo "Wechsle ins Verzeichnis <${TARGETDIR}/oplist/doc>"
    if [ ${DEBUG} -eq 0 ]; then
        cd ${TARGETDIR}/oplist/doc
    fi

    if [ ${WITH_DB_MODEL} -eq 1 ]; then
        if [ ${UPDATEYES} -eq 0 ]; then
            echo "Wir installieren das Datenmodell"
                if [ ${DEBUG} -eq 0 ]; then
                # Datenmodell ausfuehren
                if [ ${DBTYPE} = "oracle" ]; then
                    sqlplus ${SCHEMA}/${SCHEMA}@${SID} @create-oracle-db-model-oplist.sql
                    #sqlplus ${SCHEMA}/${SCHEMA}@${SID} @update-oracle-db-model-oplist.sql
                    sqlplus ${SCHEMA}/${SCHEMA}@${SID} @merge-oracle-base-entries-oplist.sql

                    # setze die installierten Parameter direkt in die Konfigurationsdatei
                    echo " " >> ${TARGETDIR}/oplist/conf/oplist.ini
                    echo "; ####------------------------------------------------------------------" >> ${TARGETDIR}/oplist/conf/oplist.ini
                    echo "; # aktuell sind mysql oder oracle moeglich, je nachdem sind dann die" >> ${TARGETDIR}/oplist/conf/oplist.ini
                    echo "; # nachfolgenden DB-Settings zu befuellen" >> ${TARGETDIR}/oplist/conf/oplist.ini
                    echo "usedb=\"oracle\";" >> ${TARGETDIR}/oplist/conf/oplist.ini
                    echo "; ####------------------------------------------------------------------" >> ${TARGETDIR}/oplist/conf/oplist.ini
                    echo "; # ORACLE" >> ${TARGETDIR}/oplist/conf/oplist.ini
                    echo "oracleServiceName = \"${SID}\";" >> ${TARGETDIR}/oplist/conf/oplist.ini
                    echo "oracleUsername = \"${SCHEMA}\";" >> ${TARGETDIR}/oplist/conf/oplist.ini
                    echo "oraclePassword = \"${SCHEMA}\";" >> ${TARGETDIR}/oplist/conf/oplist.ini
                else
                    mysql -h ${DBHOST} -u ${SCHEMA} -p ${PASSWD} ${SID} < create-mysql-db-model-oplist.sql
                    #mysql -f -h ${DBHOST} -u${SCHEMA} -p${PASSWD} ${SID} < update-mysql-db-model-oplist.sql
                    mysql -f -h ${DBHOST} -u${SCHEMA} -p${PASSWD} ${SID} < merge-mysql-base-entries-oplist.sql

                    # setze die installierten Parameter direkt in die Konfigurationsdatei
                    echo " " >> ${TARGETDIR}/oplist/conf/oplist.ini
                    echo "; ####------------------------------------------------------------------" >> ${TARGETDIR}/oplist/conf/oplist.ini
                    echo "; # aktuell sind mysql oder oracle moeglich, je nachdem sind dann die" >> ${TARGETDIR}/oplist/conf/oplist.ini
                    echo "; # nachfolgenden DB-Settings zu befuellen" >> ${TARGETDIR}/oplist/conf/oplist.ini
                    echo "usedb=\"mysql\";" >> ${TARGETDIR}/oplist/conf/oplist.ini
                    echo "; ####------------------------------------------------------------------" >> ${TARGETDIR}/oplist/conf/oplist.ini
                    echo "; # MYSQL" >> ${TARGETDIR}/oplist/conf/oplist.ini
                    echo "sqlHost = \"${DBHOST}\";" >> ${TARGETDIR}/oplist/conf/oplist.ini
                    echo "mySqlDB = \"${SID}\";" >> ${TARGETDIR}/oplist/conf/oplist.ini
                    echo "sqlUser = \"${SCHEMA}\";" >> ${TARGETDIR}/oplist/conf/oplist.ini
                    echo "sqlPass = \"${PASSWD}\";" >> ${TARGETDIR}/oplist/conf/oplist.ini
                fi
            fi
        fi
    fi

    exit 0
}

main $*
